//
//  OPDatabase.h
//  ApplicationFrameworks
//
//  Created by Rick Fillion on 2015-04-09.
//
//

#import <Foundation/Foundation.h>
#import <OnePasswordCore/OnePasswordCore.h>
#import <OnePasswordDataModel/OnePasswordDataModel.h>

@class OP4Database;
@class B5Database;
@class OPUnionDatabaseFacade;

@class B5UserAccount;
@class B5PersonalKey;
@class B5EncryptedUserKeySet;
@class B5VaultAccess;

NS_ASSUME_NONNULL_BEGIN

/// Requires subclassing
@interface OPDatabase : NSObject

+ (OP4Database *)activeOP4Database;
+ (void)setActiveOP4Database:(nullable OP4Database *)database;

+ (B5Database *)activeB5Database;
+ (void)setActiveB5Database:(nullable B5Database *)database;

+ (OPUnionDatabaseFacade *)activeUnionDatabase;

+ (NSArray *)allProfiles; // Returns the combination of profiles from OP4Database & B5Database & OPUnionDatabaseFacade
+ (NSArray *)allProfilesExcludingUnionFacade; // Returns the combination of profiles from OP4Database & B5Database
+ (NSArray *)allProfileGroups; // returns the combination of profile groups from OP4Database and B5Database and OPUnionDatabaseFacade
+ (NSArray *)allProfileGroupsExcludingUnionFacade; // returns the combination of profile groups from OP4Database and B5Database
+ (NSArray *)allUnlockedProfiles; // Returns unlocked profiles from OP4Database & B5Database

+ (void)lockAllProfiles;

+ (id<OPGenericProfile>)primaryProfile; // Returns OP4's primary profile, or nil

+ (nullable id<OPGenericProfile>)activeProfile;
+ (void)setActiveProfile:(nullable id<OPGenericProfile>)activeProfile;

+ (id<OPGenericProfile>)defaultProfile;
+ (void)setDefaultProfile:(id<OPGenericProfile>)defaultProfile;
+ (void)clearDefaultProfilePreferenceIfUUIDMatches:(NSString *)profileUUID;
+ (id<OPGenericProfile>)profileForSelectionAfterAppLaunch;
+ (void)setProfileForSelectionAfterAppLaunch:(id<OPGenericProfile>)profileForSelection;
+ (BOOL)profileForSelectionAfterAppLaunchUsesLastActiveProfile;
+ (void)setProfileForSelectionAfterAppLaunchUsesLastActiveProfile:(BOOL)value;
+ (id<OPGenericProfile>)profileForSelectionAfterAppUnlock;
+ (void)setProfileForSelectionAfterAppUnlock:(id<OPGenericProfile>)profileForSelection;
+ (BOOL)profileForSelectionAfterAppUnlockUsesLastActiveProfile;
+ (void)setProfileForSelectionAfterAppUnlockUsesLastActiveProfile:(BOOL)value;

+ (nullable id<OPGenericProfile>)profileWithUUID:(nullable NSString *)profileUUID; // Looks in each active database for a matching db.
+ (id<OPGenericProfile>)profileForItem:(OPItem *)item; // Reconnects profile based on profileType and profileId

+ (nullable OPDatabase *)databaseForProfile:(id<OPGenericProfile>)profile;
// returns id so that you can typecase it to OPDatabase<aProtocol>. Returns nil if the active database for this profile does not support this protocol
+ (nullable id)databaseForProfile:(id<OPGenericProfile>)profile conformingToProtocol:(Protocol *)aProtocol;

/**
 * Synchronous methods.
 */
- (BOOL)openDatabaseWithError:(NSError **)error;

/**
 * Asynchronous methods.
 */
- (void)close;
- (void)clearCaches;

/**
 * The offline operation will be performed when the SQLite file is closed. The SQLite file will be reopen and caches cleared after the operation is completed.
 * If before: block is not nil and returns NO then the offline operation is not performed.
 */
- (OPDatabaseOperation *)performOfflineOperation:(BOOL (^)(OPDatabase *db, NSError **error))offlineOperation before:(nullable BOOL (^)(OPDatabase *db, BOOL *rollback))before after:(nullable void (^)(BOOL success, NSError *error))after;

- (void)performSyncTransaction:(void (^)(OPDatabase *db, BOOL *rollback))transaction;
- (void)performAsyncTransaction:(void (^)(OPDatabase *db, BOOL *rollback))transaction completion:(nullable void (^)(BOOL success))completion;
- (void)performAsyncTransaction:(void (^)(OPDatabase *db, BOOL *rollback))transaction;

- (void)performLowPriorityAsyncTransaction:(void (^)(OPDatabase *db, BOOL *rollback))transaction completion:(void (^)(BOOL success))completion;
- (void)performLowPriorityAsyncTransaction:(void (^)(OPDatabase *db, BOOL *rollback))transaction;
- (void)performLowPrioritySyncTransaction:(void (^)(OPDatabase *db, BOOL *rollback))transaction;

- (id<OPGenericProfile>)profileWithUUID:(NSString *)profileUUID; // Consider moving into OPDatabaseBasicItemSupport?
- (id<OPGenericProfile>)profileWithName:(NSString *)profileName;

// Utility Methods
- (BOOL)attachDatabase:(NSString *)databaseName atPath:(NSString *)databasePath error:(NSError **)error;
- (BOOL)detachDatabase:(NSString *)databaseName error:(NSError **)error;
- (BOOL)isDatabaseAttached:(NSString *)databaseName;

// Config
- (nullable NSString *)txSelectConfigValueForKey:(NSString *)key;
- (BOOL)txSetConfigValue:(NSString *)value forKey:(NSString *)key;

@end

// Specific features that databases could support.

@protocol OPDatabaseReadonlyProfileSupport <NSObject>

- (NSArray *)allProfiles;
- (NSArray *)profileGroups;

@end

@protocol OPDatabaseProfileSupport <OPDatabaseReadonlyProfileSupport>

- (BOOL)txDeleteProfileWithId:(OPID)profileId error:(NSError **)error;
- (BOOL)txUpdateProfile:(id<OPGenericProfile>)profile error:(NSError **)error;

@end

@protocol OPDatabaseReadonlyBasicItemSupport <NSObject>

- (BOOL)txHasItemsInProfile:(id<OPGenericProfile>)profile;
- (BOOL)txHasFavoritesInProfile:(id<OPGenericProfile>)profile;
- (id)txProfile:(nullable id<OPGenericProfile>)profile selectItemWithId:(OPID)oid loadCompletely:(BOOL)loadCompletely;
- (NSArray *)txProfile:(id<OPGenericProfile>)profile selectAllItemsWithDetails:(BOOL)withDetails includeUnknown:(BOOL)includeUnknown includeTrashed:(BOOL)includeTrashed includeTombstones:(BOOL)includeTombstones;
- (NSArray *)txProfile:(id<OPGenericProfile>)profile selectAllItemsWithDetails:(BOOL)withDetails includeTrashed:(BOOL)includeTrashed includeTombstones:(BOOL)includeTombstones;
- (id)txProfile:(id<OPGenericProfile>)profile selectItemWithUUID:(NSString *)uuid loadCompletely:(BOOL)loadCompletely;
- (NSArray *)txProfile:(id<OPGenericProfile>)profile selectItemsWithUUIDs:(NSSet *)uuids loadCompletely:(BOOL)loadCompletely;
- (NSArray *)txProfile:(id<OPGenericProfile>)profile selectItemsWithIds:(NSSet *)ids loadCompletely:(BOOL)loadCompletely;
- (NSArray *)txSelectAllItemUUIDsForProfile:(id<OPGenericProfile>)profile;
- (NSArray *)txSelectAllItemUUIDsExcludingTombstonesForProfile:(OPProfile *)profile;

- (NSArray *)txSelectItemsByTagId:(OPID)tagId;
- (NSArray *)txSelectItemsByTagId:(OPID)tagId includeDetails:(BOOL)includeDetails;
- (NSArray *)txProfile:(id<OPGenericProfile>)profile selectItemsByTagTitle:(NSString *)tagTitle includeDetails:(BOOL)includeDetails;
- (NSArray *)txSelectItemTimestampsForProfileId:(OPID)profileId;
- (NSDictionary *)txSelectItemUpdatedAtForProfileId:(OPID)profileId;

- (NSArray *)txSelectLoginsFromAllProfilesForURLString:(NSString *)URLString loadCompletely:(BOOL)loadCompletely;
- (NSArray *)txProfile:(id<OPGenericProfile>)profile selectLoginsForURLString:(NSString *)URLString loadCompletely:(BOOL)loadCompletely;
- (NSArray *)txProfile:(id<OPGenericProfile>)profile selectPasswordsForURLString:(NSString *)URLString loadCompletely:(BOOL)loadCompletely;
- (NSArray *)txProfile:(id<OPGenericProfile>)profile selectItemIdsWithURLsUnderHashVersion:(OPTimestamp)hashVersion;

- (void)txSelectUsageForItems:(NSArray *)items;

- (OPItem *)txProfile:(id<OPGenericProfile>)profile loadChangesPendingForItemId:(OPID)itemId error:(NSError **)error;

- (NSUInteger)txProfile:(id<OPGenericProfile>)profile selectTotalItemCountIncludeTrashed:(BOOL)includeTrashed;

- (NSArray *)txSelectAllTagsForProfile:(id<OPGenericProfile>)profile;

- (NSArray *)txSelectFavoritesForProfile:(id<OPGenericProfile>)profile;
- (NSArray *)txSelectFavoritesForProfile:(id<OPGenericProfile>)profile includeDetails:(BOOL)includeDetails;

- (OPItemSetForURL *)txProfile:(id<OPGenericProfile>)profile selectItemSetForURLString:(NSString *)URLString;

- (NSArray *)txSelectTrashedItemsForProfile:(id<OPGenericProfile>)profile;
- (NSArray *)txSelectTrashedItemsForProfile:(id<OPGenericProfile>)profile includeDetails:(BOOL)includeDetails;

- (NSInteger)txProfileCountTrashed:(id<OPGenericProfile>)profile;
- (NSInteger)txProfileCountTombstones:(id<OPGenericProfile>)profile;

@end

@protocol OPDatabaseBasicItemSupport <OPDatabaseReadonlyBasicItemSupport>

- (BOOL)txUpdateHashesForItemURL:(OPItemURL *)itemURL error:(NSError **)error;

- (BOOL)txUpdateItem:(OPItem *)item error:(NSError **)error;
- (BOOL)txDeleteItem:(OPItem *)item error:(NSError **)error;
- (BOOL)txDeleteItemId:(OPID)itemId error:(NSError **)error;

- (BOOL)txTagItems:(NSArray *)itemIds withTag:(OPID)tagId error:(NSError **)error;
- (BOOL)txMarkItems:(NSArray *)itemIds asFavorite:(BOOL)favorite error:(NSError **)error;
- (void)txMarkItem:(OPItem *)item asFavorite:(BOOL)favorite;
- (void)txMarkItem:(OPItem *)item asTrashed:(BOOL)trashed;

- (BOOL)txRecordUsageOfItemId:(OPID)itemId error:(NSError **)error;
- (BOOL)txUpdateUsageOfItemId:(OPID)itemId usageCounter:(NSUInteger)usageCounter lastUsedAtTimestamp:(OPTimestamp)lastUsedAt error:(NSError **)error;
- (void)txRecordUsageOfItem:(OPItem *)item;

- (BOOL)txSaveChangesPendingForItem:(OPItem *)item error:(NSError **)error;
- (BOOL)txProfile:(id<OPGenericProfile>)profile removeChangesPendingForItemId:(OPID)itemId error:(NSError **)error;

- (BOOL)txDeleteTag:(OPTag *)tag error:(NSError **)error;
- (BOOL)txReplaceTag:(OPTag *)oldTag withTag:(OPTag*)newTag error:(NSError **)error;
- (BOOL)txUpdateDisplayOrderOfFavorites:(NSArray *)list error:(NSError **)error;

@end

@protocol OPDatabaseOptionalEmptyTrashSupport <NSObject>

- (BOOL)txEmptyTrashForProfile:(id<OPGenericProfile>)profile error:(NSError **)error;

@end

@protocol OPDatabaseReadonlyCategoriesSupport <NSObject>

+ (NSArray *)defaultCategoriesForProfile:(id<OPGenericProfile>)profile;
- (OPCategory *)txProfile:(id<OPGenericProfile>)profile categoryForUUID:(NSString *)categoryUUID;

- (void)txReloadCategoriesForProfile:(id<OPGenericProfile>)profile;

- (NSArray *)txProfile:(id<OPGenericProfile>)profile selectItemsInCategoryUUID:(NSString *)categoryUUID includeDetails:(BOOL)includeDetails;

- (NSArray *)txSelectCategoriesForProfile:(id<OPGenericProfile>)profile;
- (NSArray *)txAllCategoriesForProfile:(id<OPGenericProfile>)profile;
- (NSArray *)txProfile:(id<OPGenericProfile>)profile selectCategoryListWithItemCounts:(BOOL)includeItemCounts withCategoryAll:(BOOL)includeCategoryAll;

- (OPItemList *)txProfile:(id<OPGenericProfile>)profile selectItemListForCategoryUUID:(NSString *)categoryUUID excludingItems:(NSSet *)excludedItemsByUUID;

@end

@protocol OPDatabaseCategoriesSupport <OPDatabaseReadonlyCategoriesSupport>

@required
- (BOOL)txUpdateCategory:(id<OPGenericCategory>)category error:(NSError **)error;

- (BOOL)txUpdateDisplayOrderOfCategories:(NSArray *)categoryList error:(NSError **)error;
- (BOOL)txUpdateDisplayOrderOfCategoriesById:(NSDictionary *)updatedCategories error:(NSError **)error;

@end

@protocol OPDatabaseTransactionIdentificationSupport <NSObject>

@required
// [rfillion 2015-05-13] The txSyncKeychainIdentifier property should only be set at the beginning of a transaction. Hence the tx prefix. It will only persist for the duration of the current transaction so there is no need to unset it.
// When the database posts notifications upon commiting its transaction, the user info dictionary of the notification will contain key/values based on this property.
// OPDatabaseNotificationSyncTransactionFlagKey : @(YES) if txSyncKeychainIdentifier != nil
// OPDatabaseNotificationSyncedKeychainIdentifierKey : txSyncKeychainIdentifier
// In the future it'd be nice if we renamed this to have a more generic name, as the value put in this property is not always sync, nor keychain related. For example Database Migrations populate it.
@property (nonatomic, strong) NSString * txSyncKeychainIdentifier;

@end

@protocol OPDatabaseOptionalFolderSupport <NSObject>

@required
- (BOOL)txSetFolderId:(OPID)folderOid folderUUID:(nullable NSString *)folderUUID forItem:(OPID)itemOid profileUUID:(NSString *)profileUUID error:(NSError **)error;
- (void)txUpdatePendingFolderReferencesForProfile:(id<OPGenericProfile>)profile;
- (BOOL)txSetFolderId:(OPID)folderId folderUUID:(NSString *)uuid forItems:(NSArray *)itemIds profileUUID:(NSString *)profileUUID error:(NSError **)error;

- (NSArray *)txSelectFolderTimestampsForProfileId:(OPID)profileId;

- (NSSet *)txProfile:(id<OPGenericProfile>)profile selectAllFoldersIncludeTrashed:(BOOL)includeTrashed;
- (NSArray *)txSelectPlainFolderHierarchyForProfile:(id<OPGenericProfile>)profile;
- (NSArray *)txSelectAllPlainFoldersIndentedForProfile:(id<OPGenericProfile>)profile;

- (NSArray *)txSelectFolderHierarchyForProfile:(id<OPGenericProfile>)profile;
- (NSArray *)txSelectAllFoldersIndentedForProfile:(id<OPGenericProfile>)profile;

- (NSInteger)txNumberOfPlainFoldersForProfile:(id<OPGenericProfile>)profile;
- (NSArray *)txProfile:(id<OPGenericProfile>)profile selectFolderListWithParentFolderId:(OPID)parentId;

- (nullable OPFolder *)txProfile:(id<OPGenericProfile>)profile selectFolderWithUUID:(NSString *)uuid;
- (OPFolder *)txSelectFolderWithID:(OPID)oid;
- (BOOL)txUpdateFolder:(OPFolder *)folder error:(NSError **)error;
- (BOOL)txUpdateTxTimestamp:(OPTimestamp)timestamp forFolder:(OPFolder *)folder;
- (BOOL)txUpdateDisplayOrderOfFolders:(NSArray *)folders error:(NSError **)error;
- (BOOL)txMoveFolder:(OPID)folderId toParent:(OPID)parentId profileUUID:(NSString *)profileUUID error:(NSError **)error;

- (BOOL)txTrashFolderAndContents:(OPFolder *)folder error:(NSError **)error;

- (NSArray *)txProfile:(id<OPGenericProfile>)profile selectItemsByFolderId:(OPID)folderId;
- (NSArray *)txProfile:(id<OPGenericProfile>)profile selectItemsByFolderId:(OPID)folderId includeDetails:(BOOL)includeDetails;


@end

@protocol OPDatabaseOptionalAttachmentSupport <NSObject>

@required

- (OPItemAttachment *)txSelectAttachmentWithId:(OPID)attachmentId;
- (OPItemAttachment *)txProfile:(id<OPGenericProfile>)profile selectAttachmentWithUUID:(NSString *)uuid;
- (OPItemAttachment *)txProfile:(id<OPGenericProfile>)profile selectAttachmentWithUUID:(NSString *)uuid loadCompletely:(BOOL)loadCompletely;
- (NSURL *)externalContentsURLForAttachment:(OPItemAttachment *)attachment;
- (BOOL)txUpdateItemAttachment:(OPItemAttachment *)attachment error:(NSError **)error;
- (BOOL)copyAttachment:(OPItemAttachment *)sourceAttachment toAnotherItem:(OPItem *)anotherItem error:(NSError **)error;
- (BOOL)txThereAreAttachmentsForItemWithId:(OPID)itemId;
- (NSArray *)txSelectAllExternalAttachmentsForProfileId:(OPID)profileId;
- (void)txProfile:(id<OPGenericProfile>)profile selectNumberOfAttachmentsForItems:(NSArray *)items;
- (NSArray *)txSelectItemAttachmentTimestampsForProfileId:(OPID)profileId;
- (BOOL)txBuryItemAttachment:(OPItemAttachment *)attachment error:(NSError **)error;
- (NSInteger)txProfileCountAttachments:(id<OPGenericProfile>)profile;

@end

@protocol OPDatabaseOptionalTraditionalSyncSupport <NSObject>

@required

- (OPTimestamp)txSelectMaxItemUpdatedAtForProfile:(id<OPGenericProfile>)profile;
- (OPSyncSource *)txSelectSyncSourceForProfile:(id<OPGenericProfile>)profile;

@end

@protocol OPDatabaseOptionalB5AccountSupport <NSObject>

@required
- (BOOL)txUpdateUserAccount:(B5UserAccount *)userAccount error:(NSError **)error;
- (BOOL)txDeleteUserAccountWithId:(OPID)userAccountId error:(NSError **)error;
- (BOOL)txSuspendUserAccountWithId:(OPID)oid error:(NSError **)error;
- (NSArray *)allUserAccounts;
- (B5UserAccount *)userAccountWithId:(OPID)oid;

- (B5PersonalKey *)txSelectPersonalKeyForAccountId:(OPID)accountId;
- (BOOL)txUpdatePersonalKey:(B5PersonalKey *)personalKey error:(NSError **)error;

- (NSArray *)txSelectEncryptedKeySetsForAccountId:(OPID)accountId;
- (BOOL)txUpdateEncryptedKeySet:(B5EncryptedKeySet *)keySet error:(NSError **)error;
- (BOOL)txDeleteEncryptedKeySet:(B5EncryptedKeySet *)keyset error:(NSError **)error;

- (B5VaultAccess *)txSelectVaultAccessForVaultId:(OPID)vaultId accountId:(OPID)accountId;
- (BOOL)txUpdateVaultAccess:(B5VaultAccess *)vaultAccess error:(NSError **)error;

@end

@protocol OPDatabaseOptionalB5SyncSupport <NSObject>

- (NSArray *)txSelectAllItemUUIDsOfItemsWithLocalEditsForProfile:(B5Profile *)profile buildVersion:(NSUInteger)buildVersion;
- (NSArray *)txSelectAllRejectedItemsForProfile:(B5Profile *)profile;
- (BOOL)txUpdateItem:(OPItem *)item itemVersion:(NSUInteger)itemVersion localEditCount:(NSUInteger)localEditCount changerUUID:(nullable NSString *)changerUUID error:(NSError **)error;

@end

NS_ASSUME_NONNULL_END
